import { BooleanPoint } from '/sd:boolean-point.js'; 
import { StandardLabel } from '/sd:standard-label.js'; 
import { DialogWindow } from '/sd:dialog-window.js'; 
import { BasicContainer } from '/sd:basic-container.js'; 

/**
 * class to create instance of extended BooleanPoint about additional invoking DialogWindow to set new value in modbus register 
 */
export class BooleanSetPoint extends BooleanPoint {	
	/**
	 * Constructor
	 * @param {jQuery} xmlObject 		Data loaded from XML - accept boolean points part only
	 * @param {String} parentId 		HTML ID of the object, which has to be a parent of constructing boolean point
	 * @param {Integer} itemNumber 		Number of object constructing boolean point
	 * @param {String} pointMainClass 	Name of the CSS style's class of boolean point
	 * @param {String} valueType 		Type of boolean point value - "text" if falseText and trueText represent value, "image" if images represent value
	 * @param {String} falseText 		Text, which represents false value - needed for valueType equal to "text" - if null, then "false" will be displayed
	 * @param {String} trueText 		Text, which represents true value - needed for valueType equal to "text" - if null, then "true" will be displayed
	 * @param {String} themeName 		Name of chosen theme - needed for valueType equal to "image"
	 */
	constructor(xmlObject, parentId, itemNumber = 0, pointMainClass = "BooleanSetPoint", valueType = "image", falseText = null, trueText = null, themeName) {
		super(xmlObject, parentId, itemNumber, pointMainClass, valueType, falseText, trueText, themeName);
		
		var objPointer = null;
		if(valueType == "image") {
			objPointer = this.childUnactiveImage;
			this.childActiveImage.mainObj.setAttribute("onclick", "BooleanSetPoint.ShowBooleanDialogWindow(this.id, dialogWindow, booleanSetPoints)");
		}
		else if(valueType == "text")
			objPointer = this.childValue;
		objPointer.mainObj.setAttribute("onclick", "BooleanSetPoint.ShowBooleanDialogWindow(this.id, dialogWindow, booleanSetPoints)");
		this.falseText = falseText;
		this.trueText = trueText;
		this.themeName = themeName;
	}
	
	/**
	 * Inits label text of the boolean set point
	 * @override Method in BooleanPoint
	 * @param {jQuery} xmlObject 		Data loaded from XML - accept boolean set points part only
	 * @param {Integer} num 			Bit number of BooleanSetPoint in 16-bits Holding Register
	 */
	initLabelText(pointXmlObjects, num) {
		this.childLabel.setTextValue(pointXmlObjects[num].children[0].attributes['text'].value);
	}
	
	/**
	 * Inits HTML object releated to boolean set point
	 * @override Method in BooleanPoint
	 * @param {jQuery} xmlObject 		Data loaded from XML - accept boolean set points part only
	 * @param {String} xmlMarker 		XML marker of "booleansetpoint"
	 * @param {String} pointMainClass 	Name of the CSS style's class of boolean set point
	 * @param {Integer} itemNumber 		Number of object constructing boolean set point
	 */
	initObject(xmlObject, xmlMarker, pointMainClass, itemNumber) {
		var pointXmlObjects = xmlObject;
		this.childLabel = new StandardLabel(pointXmlObjects, this.id, pointMainClass + "Label", itemNumber, this.id + "label");
		this.initLabelText(pointXmlObjects, itemNumber);
		this.childValue = new StandardLabel(pointXmlObjects, this.id, pointMainClass + "Value", itemNumber, this.id + "value");
	}
	
	/**
	 * Opens popup window (DialogWindow) for editing BooleanSetPoint
	 * @param {String} id 									ID of the boolean set point
	 * @param {DialogWindow} dialogWindow 					DialogWindow instance
	 * @param {Array(BooleanSetPoints)} booleanSetPoints 	Array of all declared BooleanSetPoints
	 */
	static ShowBooleanDialogWindow(id, dialogWindow, booleanSetPoints) {
		var i;
		var identifier = "booleanSetPointsContainer" + BasicContainer.getItemNumber(id, "booleanSetPointsContainer");
		for(i=0; i < booleanSetPoints.length; i++) {
			if(booleanSetPoints[i].getId() == identifier) {
				break;
			}
		}
		if(i < booleanSetPoints.length)
			dialogWindow.show(booleanSetPoints[i]);
		else
			console.log("BooleanSetPoint " + identifier + " was no found!");
	}
}
window.BooleanSetPoint = BooleanSetPoint;